using System;
using System.Drawing;
using System.Threading;
using System.Drawing.Imaging;
using System.Text;
using System.Windows.Forms;

using System.Runtime.InteropServices;
using SensorTechnology;


namespace miniMMLSampleViewer
{
	/// <summary>
	/// StCamera
	/// </summary>
	public class FramePerSec
	{
		private Mutex mutex;
		private uint needCount;
		private uint[] rcvTimeList;
		private uint insertIndex;
		private bool isOverWritten;

		public FramePerSec(uint needCount)
		{
			mutex = new Mutex(false);
			if(needCount < 2) needCount = 2;
			this.needCount = needCount;
			rcvTimeList = new uint[needCount];
			insertIndex = 0;
			isOverWritten = false;

			Native.timeBeginPeriod(1);
		}
		private uint GetTime()
		{
			uint Time = Native.timeGetTime();
			return(Time);
		}
		public void OnReceiveImage()
		{
			mutex.WaitOne();
			rcvTimeList[insertIndex] = GetTime();
			insertIndex++;
			if(needCount <= insertIndex)
			{
				isOverWritten = true;
				insertIndex = 0;
			}
			mutex.ReleaseMutex();
		}
		private uint GetFirstTimeIndex()
		{
			uint firstTimeIndex = 0;
			if(isOverWritten)
			{
				firstTimeIndex = insertIndex;
			}
			return(firstTimeIndex);
		}
		private uint GetLatestTimeIndex()
		{
			uint latestTimeIndex = 0;
			if(0 != insertIndex)
			{
				latestTimeIndex = insertIndex - 1;
			}
			else
			{
				if(isOverWritten)
				{
					latestTimeIndex = needCount - 1;
				}
			}
			return(latestTimeIndex);
		}

		public double GetFPS()
		{
			mutex.WaitOne();
			uint firstTimeIndex = GetFirstTimeIndex();
			uint latestTimeIndex = GetLatestTimeIndex();

			uint frameCount = 0;

			if(firstTimeIndex < latestTimeIndex)
			{
				frameCount = latestTimeIndex - firstTimeIndex;
			}
			else if(latestTimeIndex < firstTimeIndex)
			{
				frameCount = needCount - 1;
			}

			uint firstTime = rcvTimeList[firstTimeIndex];
			uint latestTime = rcvTimeList[latestTimeIndex];
			mutex.ReleaseMutex();

			double framePerSec = 0.0;
			if(0 != frameCount)
			{
				uint period = 0;
				if(firstTime <= latestTime)
				{
					period = latestTime - firstTime;
				}
				else
				{
					period = uint.MaxValue - firstTime;
					period += latestTime + 1;
				}
				if(0 != period)
				{
					framePerSec = (double)frameCount * 1000.0 / (double)period;
				}
			}
			return(framePerSec);
		}

	}
	public class BGRImageBuffer
	{
		public byte[] bgrImageBuffer;
		private uint width;
		private uint height;
		private uint stPixelFormat;
		private uint frameNo;

		public BGRImageBuffer()
		{
			bgrImageBuffer = null;
			width = 0;
			height = 0;
			stPixelFormat = 0;
			frameNo = 0;
		}
		public uint Width
		{
			get
			{
				return(width);
			}
		}

		public uint Height
		{
			get
			{
				return(height);
			}
		}
		public uint StPixelFormat
		{
			get
			{
				return(stPixelFormat);
			}
		}

		public uint FrameNo
		{
			get
			{
				return(frameNo);
			}
		}
		public uint GetBuffSize()
		{
			uint	bufferSize = width * height;
			switch(stPixelFormat)
			{
				case(StTrg.STCAM_PIXEL_FORMAT_08_MONO_OR_RAW):
					break;
				case(StTrg.STCAM_PIXEL_FORMAT_24_BGR):
					bufferSize *= 3;
					break;
				case(StTrg.STCAM_PIXEL_FORMAT_32_BGR):
					bufferSize *= 4;
					break;
			}
			return(bufferSize);
		}
		private void CreateBuffer(uint width, uint height, uint stPixelFormat)
		{
			if(
				(null == this.bgrImageBuffer) ||
				(width != this.width) ||
				(height != this.height) ||
				(stPixelFormat != this.stPixelFormat)
				)
			{
				this.width = width;
				this.height = height;
				this.stPixelFormat = stPixelFormat;

				this.bgrImageBuffer = new byte[GetBuffSize()];
			}
		}
		public void ColorInterpolation(uint frameNo, uint width, uint height, ushort colorArray, System.IntPtr pbyteRaw, byte colorInterpolationMethod, uint stPixelFormat)
		{
			CreateBuffer(width, height, stPixelFormat);
			this.frameNo = frameNo;

			//Color Interpolation
			unsafe
			{
				fixed(byte * pbyteBGR = &bgrImageBuffer[0])
				{
					StTrg.ColorInterpolation(width, height, colorArray, pbyteRaw, (System.IntPtr)pbyteBGR, colorInterpolationMethod, stPixelFormat);
				}
			}

		}
        public void BGRGamma(System.IntPtr hCamera)
        {
            //BGR Gamma

            unsafe
            {
                fixed (byte* pbyteBGR = &bgrImageBuffer[0])
                {
                    StTrg.BGRGamma(hCamera, width, height, stPixelFormat, (System.IntPtr)pbyteBGR);
                }
            }
        }
		public void HueSaturationColorMatrix(System.IntPtr hCamera)
		{
			//HueSaturationColorMatrix
			unsafe
			{
				fixed(byte * pbyteBGR = &bgrImageBuffer[0])
				{
					StTrg.HueSaturationColorMatrix(hCamera, width, height, stPixelFormat, (System.IntPtr)pbyteBGR);
				}
			}
		}
		public void Sharpness(System.IntPtr hCamera)
		{
			//Sharpness
			unsafe
			{
				fixed(byte * pbyteBGR = &bgrImageBuffer[0])
				{
					StTrg.Sharpness(hCamera, width, height, stPixelFormat, (System.IntPtr)pbyteBGR);
				}
			}
		}

		public bool SaveImageFile(string fileName)
		{
			bool result = false;
			unsafe
			{
				fixed(byte * pbyteBGR = &bgrImageBuffer[0])
				{
					result = StTrg.SaveImage(width, height, stPixelFormat, (System.IntPtr)pbyteBGR, fileName, 0);
				}
			}
			return(result);
		}
		public bool Draw(System.IntPtr hCamera, System.IntPtr hWnd)
		{
			bool result = false;
			if(null != bgrImageBuffer)
			{
				Native.SetWindowPos(hWnd, IntPtr.Zero, 0, 0, (int)width, (int)height, Native.SWP_NOMOVE | Native.SWP_NOZORDER);
				System.IntPtr hDC = Native.GetDC(hWnd);
				unsafe
				{
					fixed(byte * pbyteBGR = &bgrImageBuffer[0])
					{	
						result = StTrg.Draw(hCamera, hDC,
							0, 0, width, height,
							0, 0, width, height,
							width, height, (System.IntPtr)pbyteBGR, stPixelFormat);
					}
				}
				Native.ReleaseDC(hWnd, hDC);
			}
			return(result);
		}

	}
	
	#region Classes for IO Pin
	public class CIOPinInOut
	{
		System.IntPtr hCamera;
		public CIOPinInOut(System.IntPtr hCamera)
		{
			this.hCamera = hCamera;
		}

		public uint this[byte pinNo]
		{
			get
			{
				uint direction = 0;
				StTrg.GetIOPinDirection(hCamera, out direction);
				direction >>= (int)pinNo;
				return(direction & 0x00000001);
			}
			set
			{
				uint direction = 0;
				StTrg.GetIOPinDirection(hCamera, out direction);
				
				direction &= ~((uint)1 << (int)pinNo);
				direction |= (value & 0x00000001) << (int)pinNo;
				StTrg.SetIOPinDirection(hCamera, direction);
			}
		}
	}
	public class CIOPinMode
	{
		System.IntPtr hCamera;
		public CIOPinMode(System.IntPtr hCamera)
		{
			this.hCamera = hCamera;
		}

		public uint this[byte pinNo]
		{
			get
			{
				uint mode = 0;
				StTrg.GetIOPinMode(hCamera, pinNo, out mode);
				return(mode);
			}
			set
			{
				StTrg.SetIOPinMode(hCamera, pinNo, value);
			}
		}
	}
	public class CIOPinPolarity
	{
		System.IntPtr hCamera;
		public CIOPinPolarity(System.IntPtr hCamera)
		{
			this.hCamera = hCamera;
		}

		public uint this[byte pinNo]
		{
			get
			{
				uint polarity = 0;
				StTrg.GetIOPinPolarity(hCamera, out polarity);
				polarity >>= (int)pinNo;
				return(polarity & 0x00000001);
			}
			set
			{
				uint polarity = 0;
				StTrg.GetIOPinPolarity(hCamera, out polarity);
				
				polarity &= ~((uint)1 << (int)pinNo);
				polarity |= (value & 0x00000001) << (int)pinNo;
				StTrg.SetIOPinPolarity(hCamera, polarity);
			}
		}
	}
	public class CIOPinStatus
	{
		System.IntPtr hCamera;
		public CIOPinStatus(System.IntPtr hCamera)
		{
			this.hCamera = hCamera;
		}

		public uint this[byte pinNo]
		{
			get
			{
				uint status = 0;
				StTrg.GetIOPinStatus(hCamera, out status);
				status >>= (int)pinNo;
				return(status & 0x00000001);
			}
			set
			{
				uint status = 0;
				StTrg.GetIOPinStatus(hCamera, out status);
				
				status &= ~((uint)1 << (int)pinNo);
				status |= (value & 0x00000001) << (int)pinNo;
				StTrg.SetIOPinStatus(hCamera, status);
			}
		}
	}
	public class CSwStatus
	{
		System.IntPtr hCamera;
		public CSwStatus(System.IntPtr hCamera)
		{
			this.hCamera = hCamera;
		}

		public uint this[byte pinNo]
		{
			get
			{
				uint status = 0;
				StTrg.GetSwStatus(hCamera, out status);
				status >>= (int)pinNo;
				return(status & 0x00000001);
			}
		}
	}


	#endregion
	#region TimeOut
	public class CTimeOut
	{
		System.IntPtr hCamera;
		public CTimeOut(System.IntPtr hCamera)
		{
			this.hCamera = hCamera;
		}
		public uint this[uint timeOutType]
		{
			get
			{
				uint Value = 0;
				StTrg.GetTimeOut(hCamera, timeOutType, out Value);
				return(Value);
			}
			set
			{
				StTrg.StopTransfer(hCamera);
				StTrg.SetTimeOut(hCamera, timeOutType, value);
				StTrg.StartTransfer(hCamera);
				
			}
		}
	}
	#endregion
	#region Timing
	public class CTriggerTiming
	{
		System.IntPtr hCamera;
		public CTriggerTiming(System.IntPtr hCamera)
		{
			this.hCamera = hCamera;
		}
		public uint this[uint triggerTimingType]
		{
			get
			{
				uint Value = 0;
				StTrg.GetTriggerTiming(hCamera, triggerTimingType, out Value);
				return(Value);
			}
			set
			{
				StTrg.SetTriggerTiming(hCamera, triggerTimingType, value);
			}
		}
	}
	public class CTriggerTimingText
	{
		System.IntPtr hCamera;
		public CTriggerTimingText(System.IntPtr hCamera)
		{
			this.hCamera = hCamera;
		}
		public string this[uint triggerTimingType]
		{
			get
			{
				uint setValue = 0;
				StTrg.GetTriggerTiming(hCamera, triggerTimingType, out setValue);

				uint ClockMode = 0;
				uint PixelClock = 0;
				StTrg.GetClock(hCamera, out ClockMode, out PixelClock);


				double sec = 0.0;
				if(0 != PixelClock)
				{
					if(StTrg.STCAM_TRIGGER_TIMING_READOUT_DELAY == triggerTimingType)
					{						
						ushort TotalLine = 0;
						ushort ClockPerLine = 0;
						StTrg.GetFrameClock(hCamera, out TotalLine, out ClockPerLine);
						sec = setValue * ClockPerLine  / (double)PixelClock;		
					}
					else
					{
						sec = setValue  / (double)PixelClock;	
					}	
				}

				if(1.0 < sec)
				{
					return(string.Format("{0:N1}s", sec));
				}
				double mSec = sec * 1000.0;
				if(1.0 < mSec)
				{
					return(string.Format("{0:N1}ms", mSec));
				}
				double uSec = mSec * 1000.0;
				if(1.0 < uSec)
				{
					return(string.Format("{0:N1}us", uSec));
				}
				double nSec = uSec * 1000.0;
				return(string.Format("{0:N1}ns", nSec));

			}
		}
	}
	#endregion

    #region Gamma
    public class CGammaValue
    {
        System.IntPtr hCamera;
        public CGammaValue(System.IntPtr hCamera)
        {
            this.hCamera = hCamera;
        }
        public ushort this[byte gammaTarget]
        {
            get
            {
                byte byteGammaMode = 0;
                ushort wGammaValue = 0;
                short shtBrightness = 0;
                byte byteContrast = 0;
                StTrg.GetGammaModeEx(hCamera, gammaTarget, out byteGammaMode, out wGammaValue, out shtBrightness, out byteContrast, IntPtr.Zero);
                return (wGammaValue);
            }
            set
            {
                byte byteGammaMode = 0;
                ushort wGammaValue = 0;
                short shtBrightness = 0;
                byte byteContrast = 0;
                StTrg.GetGammaModeEx(hCamera, gammaTarget, out byteGammaMode, out wGammaValue, out shtBrightness, out byteContrast, IntPtr.Zero);
                wGammaValue = value;
                StTrg.SetGammaModeEx(hCamera, gammaTarget, byteGammaMode, wGammaValue, shtBrightness, byteContrast, IntPtr.Zero);
            }
        }
    }
    public class CGammaMode
    {
        System.IntPtr hCamera;
        public CGammaMode(System.IntPtr hCamera)
        {
            this.hCamera = hCamera;
        }
        public byte this[byte gammaTarget]
        {
            get
            {
                byte byteGammaMode = 0;
                ushort wGammaValue = 0;
                short shtBrightness = 0;
                byte byteContrast = 0;
                StTrg.GetGammaModeEx(hCamera, gammaTarget, out byteGammaMode, out wGammaValue, out shtBrightness, out byteContrast, IntPtr.Zero);
                return (byteGammaMode);
            }
            set
            {
                byte byteGammaMode = 0;
                ushort wGammaValue = 0;
                short shtBrightness = 0;
                byte byteContrast = 0;
                StTrg.GetGammaModeEx(hCamera, gammaTarget, out byteGammaMode, out wGammaValue, out shtBrightness, out byteContrast, IntPtr.Zero);
                byteGammaMode = value;
                StTrg.SetGammaModeEx(hCamera, gammaTarget, byteGammaMode, wGammaValue, shtBrightness, byteContrast, IntPtr.Zero);
            }
        }
    }
    #endregion
	public class StCamera
	{
		//
		static readonly IntPtr INVALID_HANDLE_VALUE = new IntPtr(unchecked(-1));

		//Camera Info
		private System.IntPtr hCamera;
		
		//IO
		public CIOPinInOut IOPinInOut;
		public CIOPinMode	IOPinMode;
		public CIOPinPolarity	IOPinPolarity;
		public CIOPinStatus	IOPinStatus;
		public CSwStatus SwStatus;

		//Time Out
		public CTimeOut TimeOut;
		public CTriggerTiming TriggerTiming;
		public CTriggerTimingText TriggerTimingText;

		//FPS
		private FramePerSec framePerSec;

        //Gamma
        public CGammaValue GammaValue;
        public CGammaMode GammaMode;

        //SDK Version
        private uint m_dwFileVersionMS;
        private uint m_dwFileVersionLS;
        private uint m_dwProductVersionMS;
        private uint m_dwProductVersionLS;
		//
		private ushort usbVendorID;
		private ushort usbProductID;
		private ushort fpgaVersion;
		private ushort firmwareVersion;
		private ushort colorArray;

		//Buffer
		private Mutex receivingBGRMutex;
		private Mutex displayBGRMutex;

		private BGRImageBuffer receivingBGRImage;
		private BGRImageBuffer displayBGRImage;

		//Setting
		private uint stPixelFormat;
		private byte colorInterpolationMethod;
		private uint noiseReductionMode;
		private byte mirrorMode;
		private byte rotationMode;

		//Callback
		private StTrg.funcTransferEndCallback transferEndCallback;
		private StTrg.funcExposureEndCallback exposureEndCallback;
		private StTrg.funcRcvErrorCallback rcvErrorCallback;

		public delegate void DisplayImageCallback(uint dwFrameNo, uint Width, uint Height, uint StPixelFormat, byte[] BGRImage);
		private Mutex displayImageCallbackMutex;
		private DisplayImageCallback displayImageCallback;

		public delegate void SettingUpdateCallback();
		public SettingUpdateCallback OnSettingUpdate;


		//Dropped Frame Count
		private uint dropFrameCount;

		//Setting Dialog
		private SettingForm settingForm;


		//Auto StartStop
		private int m_AutoTriggerDueTime;
		private bool m_bAutoTrigger;
		System.Threading.Timer	m_objTimer;


		public StCamera(System.IntPtr hCamera)
		{
			//Camera Info
			this.hCamera = hCamera;

            bool bReval = StTrg.GetDllVersion(out m_dwFileVersionMS, out m_dwFileVersionLS, out m_dwProductVersionMS, out m_dwProductVersionLS);

			//IO
			IOPinInOut = new CIOPinInOut(hCamera);
			IOPinMode = new CIOPinMode(hCamera);
			IOPinPolarity = new CIOPinPolarity(hCamera);
			IOPinStatus = new CIOPinStatus(hCamera);
			SwStatus = new CSwStatus(hCamera);

			//Time Out
			TimeOut = new CTimeOut(hCamera);
			TriggerTiming = new CTriggerTiming(hCamera);
			TriggerTimingText = new CTriggerTimingText(hCamera);

			//FPS
			framePerSec = new FramePerSec(100);
            
            //Gamma
            GammaValue = new CGammaValue(hCamera);
            GammaMode = new CGammaMode(hCamera);

			usbVendorID = 0;
			usbProductID = 0;
			fpgaVersion = 0;
			firmwareVersion = 0;
			StTrg.GetCameraVersion(hCamera, out usbVendorID, out usbProductID, out fpgaVersion, out firmwareVersion);

			colorArray = 0;
			StTrg.GetColorArray(hCamera, out colorArray);

			//BGR Image Buffer
			receivingBGRMutex = new Mutex(false);
			displayBGRMutex = new Mutex(false);

			receivingBGRImage = new BGRImageBuffer();
			displayBGRImage = new BGRImageBuffer();

			//Setting
			stPixelFormat = StTrg.STCAM_PIXEL_FORMAT_08_MONO_OR_RAW;
			if(StTrg.STCAM_COLOR_ARRAY_MONO != colorArray)
			{	
				stPixelFormat = StTrg.STCAM_PIXEL_FORMAT_24_BGR;
			}

			colorInterpolationMethod = StTrg.STCAM_COLOR_INTERPOLATION_BILINEAR;
			noiseReductionMode = StTrg.STCAM_NR_OFF;

			mirrorMode = StTrg.STCAM_MIRROR_OFF;
			rotationMode = StTrg.STCAM_ROTATION_OFF;

			//Callback 
			displayImageCallbackMutex = new Mutex(false);
			displayImageCallback = null;

			transferEndCallback = new StTrg.funcTransferEndCallback(OnTransferEnd);
			StTrg.SetTransferEndCallback(hCamera, transferEndCallback, IntPtr.Zero);

			exposureEndCallback = new StTrg.funcExposureEndCallback(OnExposureEnd);
			StTrg.SetExposureEndCallback(hCamera, exposureEndCallback, IntPtr.Zero);

			rcvErrorCallback = new StTrg.funcRcvErrorCallback(OnRcvError);
			StTrg.SetRcvErrorCallback(hCamera, rcvErrorCallback, IntPtr.Zero);

			OnSettingUpdate = null;
			

			//Dropped Frame Count
			dropFrameCount = 0;
			
			//Setting Form
			settingForm = null;


			//StartStop
			m_bAutoTrigger = false;
			m_AutoTriggerDueTime = 1000;
			m_objTimer = new System.Threading.Timer(new TimerCallback(timerAutoTrigger), null, Timeout.Infinite, Timeout.Infinite);

			StartTransfer();
		}

		public void timerAutoTrigger(object o)
		{
			SoftSubTrigger();
		}

		public uint StPixelFormat
		{
			get
			{
				return(stPixelFormat);
			}
			set
			{
				stPixelFormat = value;
			}
		}
		public byte ColorInterpolationMode
		{
			get
			{
				return(colorInterpolationMethod);
			}
			set
			{
				colorInterpolationMethod = value;
			}
		}
		public byte MirrorMode
		{
			get
			{
				return(mirrorMode);
			}
			set
			{
				mirrorMode = value;
			}
		}
		public byte RotationMode
		{
			get
			{
				return(rotationMode);
			}
			set
			{
				rotationMode = value;
			}
		}
		public uint NoiseRedutionMode
		{
			get
			{
				return(noiseReductionMode);
			}
			set
			{
				noiseReductionMode = value;
			}
		}
		
		#region Open/Close Camera Method
		public static int OpenCameraList(int MaxCameraCount, System.IntPtr[] ahCameras, uint[] aCameraID, string[] astrCameraName)
		{

			StringBuilder sbTmpCameraName = new StringBuilder(256);
			int ExistCameraCount = 0;

			for(int i = 0; i < MaxCameraCount; i++)
			{

				ahCameras[i] = StTrg.Open();
				if(INVALID_HANDLE_VALUE == ahCameras[i])
				{
					break;
				}
				else
				{
					ExistCameraCount++;

					StTrg.ReadCameraUserID(ahCameras[i], out aCameraID[i], sbTmpCameraName, (uint)(sbTmpCameraName.Capacity));
					astrCameraName[i] = sbTmpCameraName.ToString();

					System.Console.WriteLine(astrCameraName[i]);
				}
			}
			return(ExistCameraCount);

		}
		public static  void CloseCameraList(int CameraCount, System.IntPtr[] ahCameras, System.IntPtr hExceptCamera)
		{
			for(int i = 0; i < CameraCount; i++)
			{
				if(ahCameras[i] != hExceptCamera)
				{
					StTrg.Close(ahCameras[i]);
				}
			}
		}
		public static StCamera Open()
		{
			SelectCameraForm selectCameraForm = new SelectCameraForm();
			DialogResult dlgResult = selectCameraForm.ShowDialog();
			System.IntPtr hCamera = selectCameraForm.SelectCameraHandle;
			selectCameraForm.Dispose();
			if(DialogResult.OK != dlgResult)
			{
				return(null);
			}

			StCamera stCamera = new StCamera(hCamera);

			return(stCamera);
		}
		public void Close()
		{
			if(IsOpenned())
			{
				StopTransfer();
				StTrg.Close(hCamera);
				hCamera = IntPtr.Zero;
			}
		}
		#endregion

		public void SetDisplayImageCallback(DisplayImageCallback callback)
		{
			displayImageCallbackMutex.WaitOne();
			displayImageCallback = callback;
			displayImageCallbackMutex.ReleaseMutex();
		}


		public void OnExposureEnd(System.IntPtr hCamera, uint dwFrameNo, System.IntPtr lpContext)
		{


		}

		private byte[] m_pbyteTmpRaw = null;
		public void OnTransferEnd(System.IntPtr hCamera, uint dwFrameNo, uint dwWidth, uint dwHeight, ushort wColorArray, System.IntPtr pvRaw, System.IntPtr lpContext)
		{
			//for FPS
			framePerSec.OnReceiveImage();

			if (TransferBitsPerPixel != StTrg.STCAM_TRANSFER_BITS_PER_PIXEL_08)
			{
				uint nSize = dwWidth * dwHeight;
				if (
					(m_pbyteTmpRaw == null) ||
					(m_pbyteTmpRaw.GetLength(0) < nSize)
				)
				{
					m_pbyteTmpRaw = new byte[nSize];
				}
				System.Runtime.InteropServices.GCHandle gch = System.Runtime.InteropServices.GCHandle.Alloc(m_pbyteTmpRaw, System.Runtime.InteropServices.GCHandleType.Pinned);
				System.IntPtr pvTmpRaw = gch.AddrOfPinnedObject();
				StTrg.ConvTo8BitsImage(dwWidth, dwHeight, TransferBitsPerPixel, pvRaw, pvTmpRaw);

				ImageProcessing(dwFrameNo, dwWidth, dwHeight, wColorArray, pvTmpRaw);
				gch.Free();

			}
			else
			{
				ImageProcessing(dwFrameNo, dwWidth, dwHeight, wColorArray, pvRaw);
			}
		}
		private void ImageProcessing(uint dwFrameNo, uint dwWidth, uint dwHeight, ushort wColorArray, System.IntPtr pvRaw)
		{
			//Noise Reduction
			StTrg.NoiseReduction(hCamera, noiseReductionMode, dwWidth, dwHeight, wColorArray, pvRaw);

			//White Balance 
			StTrg.WhiteBalance(hCamera, pvRaw);

            //Color Gamma
			StTrg.RawColorGamma(hCamera, dwWidth, dwHeight, wColorArray, pvRaw);

			//Mirror Rotation
			StTrg.MirrorRotation(mirrorMode, rotationMode, ref dwWidth, ref dwHeight, ref wColorArray, pvRaw);

			receivingBGRMutex.WaitOne();

			//Color Interpolation
			receivingBGRImage.ColorInterpolation(dwFrameNo, dwWidth, dwHeight, wColorArray, pvRaw, colorInterpolationMethod, stPixelFormat);

            //BGRGamma
            receivingBGRImage.BGRGamma(hCamera);

			//Hue/Saturation
			receivingBGRImage.HueSaturationColorMatrix(hCamera);

			//Sharpness
			receivingBGRImage.Sharpness(hCamera);

			displayBGRMutex.WaitOne();

			//Swap Receive Buffer To Show buffer
			BGRImageBuffer tmpBuffer = receivingBGRImage;
			receivingBGRImage = displayBGRImage;
			displayBGRImage = tmpBuffer;


			//BGRImageCallback
			displayImageCallbackMutex.WaitOne();
			if(null != displayImageCallback)
			{
				displayImageCallback(displayBGRImage.FrameNo, displayBGRImage.Width, displayBGRImage.Height, displayBGRImage.StPixelFormat, displayBGRImage.bgrImageBuffer);
			}
			displayImageCallbackMutex.ReleaseMutex();
		
			displayBGRMutex.ReleaseMutex();

			receivingBGRMutex.ReleaseMutex();

		}
		public void OnRcvError(System.IntPtr hCamera, uint dwErrorCode, System.IntPtr lpContext)
		{
			//Counter
			if(
				(StTrg.ERR_EXPOSURE_END_DROPPED == dwErrorCode) ||
				(StTrg.ERR_IMAGE_DATA_DROPPED == dwErrorCode)
				)
			{
				dropFrameCount++;	
			}

			//Access Denied
			if(Native.ERROR_ACCESS_DENIED == dwErrorCode)
			{
				//m_bTransferFg = FALSE;
			}
		}
		public double GetFPS()
		{
			return(framePerSec.GetFPS());
		}
		//Dropped Frame Count
		public uint GetDroppedFrameCount()
		{
			return(dropFrameCount);
		}
		public ushort USBVID
		{
			get
			{
				return(usbVendorID);
			}
		}
		public ushort USBPID
		{
			get
			{
				return(usbProductID);
			}
		}
		public string CameraType
		{
			get
			{
				string result = "";
				switch(usbProductID)
				{
					case(StTrg.STCAM_USBPID_STC_B33USB):
						result = "STC-B33USB";
						break;
					case(StTrg.STCAM_USBPID_STC_C33USB):
						result = "STC-C33USB";
						break;
					case(StTrg.STCAM_USBPID_STC_B83USB):
						result = "STC-B83USB";
						break;
					case(StTrg.STCAM_USBPID_STC_C83USB):
						result = "STC-C83USB";
						break;
					case(StTrg.STCAM_USBPID_STC_TB33USB):
						result = "STC-TB33USB";
						break;
					case(StTrg.STCAM_USBPID_STC_TC33USB):
						result = "STC-TC33USB";
						break;
					case(StTrg.STCAM_USBPID_STC_TB83USB):
						result = "STC-TB33USB";
						break;
					case(StTrg.STCAM_USBPID_STC_TC83USB):
						result = "STC-TC83USB";
                        break;
                    case (StTrg.STCAM_USBPID_STC_TB133USB):
                        result = "STC-TB133USB";
                        break;
                    case (StTrg.STCAM_USBPID_STC_TC133USB):
                        result = "STC-TC133USB";
                        break;
					case(StTrg.STCAM_USBPID_STC_TB152USB):
						result = "STC-TB152USB";
						break;
					case(StTrg.STCAM_USBPID_STC_TC152USB):
						result = "STC-TC152USB";
						break;
					case(StTrg.STCAM_USBPID_STC_TB202USB):
						result = "STC-TB202USB";
						break;
					case(StTrg.STCAM_USBPID_STC_TC202USB):
						result = "STC-TC202USB";
						break;
					case (StTrg.STCAM_USBPID_STC_MB33USB):
						result = "STC-MB33USB";
						break;
					case (StTrg.STCAM_USBPID_STC_MC33USB):
						result = "STC-MC33USB";
						break;
					case (StTrg.STCAM_USBPID_STC_MB83USB):
						result = "STC-MB83USB";
						break;
					case (StTrg.STCAM_USBPID_STC_MC83USB):
						result = "STC-MC83USB";
						break;
					case (StTrg.STCAM_USBPID_STC_MB133USB):
						result = "STC-MB133USB";
						break;
					case (StTrg.STCAM_USBPID_STC_MC133USB):
						result = "STC-MC133USB";
						break;
					case (StTrg.STCAM_USBPID_STC_MB152USB):
						result = "STC-MB152USB";
						break;
					case (StTrg.STCAM_USBPID_STC_MC152USB):
						result = "STC-MC152USB";
						break;
					case (StTrg.STCAM_USBPID_STC_MB202USB):
						result = "STC-MB202USB";
						break;
					case (StTrg.STCAM_USBPID_STC_MC202USB):
						result = "STC-MC202USB";
						break;
					case (StTrg.STCAM_USBPID_APBWVUSB_LED):
						result = "APBWVUSB-LED";
						break;
					case (StTrg.STCAM_USBPID_APCWVUSB_LED):
						result = "APCWVUSB-LED";
						break;
				}
				return(result);
			}
		}
		public ushort FPGAVersion
		{
			get
			{
				return(fpgaVersion);
			}
		}
		public ushort FirmwareVersion
		{
			get
			{
				return(firmwareVersion);
			}
		}
        public string SDKVersion
        {
            get{
                return(string.Format("{0:X}.{1:X}.{2:X}.{3:X}", 
                    m_dwFileVersionMS >> 16, 
                    m_dwFileVersionMS & 0xFFFF,
                    m_dwFileVersionLS >> 16,
                    m_dwFileVersionLS & 0xFFFF));
            }
        }
		public ushort ColorArray
		{
			get
			{
				return(colorArray);
			}
		}
		
		public bool HasDisplayImage()
		{
	
			displayBGRMutex.WaitOne();
			bool bExist = (null != displayBGRImage.bgrImageBuffer);
			displayBGRMutex.ReleaseMutex();
			return(bExist);
		}
		public bool HasTriggerFunction()
		{
			bool result = true;
			switch(usbProductID)
			{
				case(StTrg.STCAM_USBPID_STC_B33USB):
				case(StTrg.STCAM_USBPID_STC_C33USB):
				case(StTrg.STCAM_USBPID_STC_B83USB):
				case(StTrg.STCAM_USBPID_STC_C83USB):
					result = false;
					break;
			}
			return(result);
		}


		public bool HasDigitalGainFunction()
		{
			return(HasTriggerFunction());
		}
		public bool HasBinningPartialFunction()
		{
			return(HasTriggerFunction());
		}
		public bool HasStartStopExposureFunction()
		{
			bool result = false;
			StTrg.HasFunction(hCamera, StTrg.STCAM_CAMERA_FUNCTION_STARTSTOP, out result);
			return(result);
		}
		public bool HasCameraMemoryFunction()
		{
			bool result = false;
			StTrg.HasFunction(hCamera, StTrg.STCAM_CAMERA_FUNCTION_MEMORY, out result);
			return(result);
		}
		public bool HasTriggerMaskFunction()
		{
			bool result = false;
			StTrg.HasFunction(hCamera, StTrg.STCAM_CAMERA_FUNCTION_TRIGGER_MASK, out result);
			return (result);
		}

		public bool HasTriggerThroughFunction()
		{
			bool result = false;
			StTrg.HasFunction(hCamera, StTrg.STCAM_CAMERA_FUNCTION_PHOTOCOUPLER, out result);
			return (!result);
		}
		public bool HasChangeIOFunction()
		{
			bool result = false;
			StTrg.HasFunction(hCamera, StTrg.STCAM_CAMERA_FUNCTION_IO_CHANGE_DIRECTION, out result);
			return(result);
		}
		public bool HasLEDFunction()
		{
			bool result = false;
			StTrg.HasFunction(hCamera, StTrg.STCAM_CAMERA_FUNCTION_LED, out result);
			return(result);
		}
		public int CDSGainType()
		{
			int nType = 0;

			bool result = false;
			StTrg.HasFunction(hCamera, StTrg.STCAM_CAMERA_FUNCTION_CDS_GAIN_TYPE, out result);
			if (result)
			{
				nType = 1;
			}
			else if ((USBPID == StTrg.STCAM_USBPID_APBWVUSB_LED) || (USBPID == StTrg.STCAM_USBPID_APCWVUSB_LED))
			{
				nType = 9;
			}
			return (nType);
		}
		public bool IsOpenned()
		{
			return(hCamera != IntPtr.Zero);
		}
		public bool SaveImage(string fileName)
		{
			bool result = false;
			if(HasDisplayImage())
			{
				displayBGRMutex.WaitOne();
				result = displayBGRImage.SaveImageFile(fileName);
				displayBGRMutex.ReleaseMutex();
			}
			return(result);
		}
		public bool Draw(System.IntPtr hWnd, out uint frameNo)
		{
			bool result = false;
			frameNo = 0;
			if(IsOpenned())
			{
				displayBGRMutex.WaitOne();
				result = displayBGRImage.Draw(hCamera, hWnd);
				frameNo = displayBGRImage.FrameNo;
				displayBGRMutex.ReleaseMutex();
			}
			return(result);
		}
		public void ShowSettingDlg()
		{
			if(IsOpenned())
			{
				if(null == settingForm)
				{
					settingForm = new SettingForm(this);
					settingForm.FormClosed += new FormClosedEventHandler(settingForm_FormClosed);
				}
				settingForm.Show();
				settingForm.Activate();
			}
		}

		void settingForm_FormClosed(object sender, FormClosedEventArgs e)
		{
			settingForm = null;
		}
		public void ShowCameraIDDlg()
		{
			if(IsOpenned())
			{
				SetCameraIDForm dlg = new SetCameraIDForm();
				dlg.CameraName = this.CameraUserName;
				dlg.CameraNo = this.CameraUserNo;
			
				if(DialogResult.OK == dlg.ShowDialog())
				{
					this.CameraUserName = dlg.CameraName;
					this.CameraUserNo = dlg.CameraNo;

				}
				dlg.Dispose();
			}
		}
		#region Transfer
		public bool StartTransfer()
		{
			bool result = false;
			if(IsOpenned())
			{
				result = StTrg.StartTransfer(hCamera);
			}
			return(result);
		}
		public bool StopTransfer()
		{
			bool result = false;
			if(IsOpenned())
			{
				result = StTrg.StopTransfer(hCamera);
			}
			return(result);
		}
		#endregion

		//---------------------------------------------------------------------------
		#region ScanMode
		public ushort ScanMode
		{
			set
			{
				if(IsOpenned())
				{
					StopTransfer();
					ushort scanMode = 0;
					uint offsetX = 0;
					uint offsetY = 0;
					uint width = 0;
					uint height = 0;
					StTrg.GetScanMode(hCamera, out scanMode, out offsetX, out offsetY, out width, out height);
					scanMode = value;
					StTrg.SetScanMode(hCamera, scanMode, offsetX, offsetY, width, height);
					StartTransfer();
				}
			}
			get
			{
				ushort Value = 0;
				if(IsOpenned())
				{
					ushort scanMode = 0;
					uint offsetX = 0;
					uint offsetY = 0;
					uint width = 0;
					uint height = 0;
					StTrg.GetScanMode(hCamera, out scanMode, out offsetX, out offsetY, out width, out height);
					Value = scanMode;
				}
				return(Value);
			}
		}

		public uint AOIOffsetY
		{
			set
			{
				if (IsOpenned())
				{
					StopTransfer();
					ushort scanMode = 0;
					uint offsetX = 0;
					uint offsetY = 0;
					uint width = 0;
					uint height = 0;
					StTrg.GetScanMode(hCamera, out scanMode, out offsetX, out offsetY, out width, out height);
					offsetY = value;
					StTrg.SetScanMode(hCamera, scanMode, offsetX, offsetY, width, height);
					StartTransfer();
				}
			}
			get
			{
				uint Value = 0;
				if (IsOpenned())
				{
					ushort scanMode = 0;
					uint offsetX = 0;
					uint offsetY = 0;
					uint width = 0;
					uint height = 0;
					StTrg.GetScanMode(hCamera, out scanMode, out offsetX, out offsetY, out width, out height);
					Value = offsetY;
				}
				return (Value);
			}
		}
		public uint AOIHeight
		{
			set
			{
				if (IsOpenned())
				{
					StopTransfer();
					ushort scanMode = 0;
					uint offsetX = 0;
					uint offsetY = 0;
					uint width = 0;
					uint height = 0;
					StTrg.GetScanMode(hCamera, out scanMode, out offsetX, out offsetY, out width, out height);
					height = value;
					StTrg.SetScanMode(hCamera, scanMode, offsetX, offsetY, width, height);
					StartTransfer();
				}
			}
			get
			{
				uint Value = 0;
				if (IsOpenned())
				{
					ushort scanMode = 0;
					uint offsetX = 0;
					uint offsetY = 0;
					uint width = 0;
					uint height = 0;
					StTrg.GetScanMode(hCamera, out scanMode, out offsetX, out offsetY, out width, out height);
					Value = height;
				}
				return (Value);
			}
		}
		public bool GetScanMode(out ushort pwScanMode, out uint pdwOffsetX, out uint pdwOffsetY, out uint pdwWidth, out uint pdwHeight)
		{
			bool result = false;
			pwScanMode = 0;
			pdwOffsetX = 0;
			pdwOffsetY = 0;
			pdwWidth = 0;
			pdwHeight = 0;
			
			if(IsOpenned())
			{
				result = StTrg.GetScanMode(hCamera, out pwScanMode, out pdwOffsetX, out pdwOffsetY, out pdwWidth, out pdwHeight);
			}
			return(result);
		}
		#endregion

		//---------------------------------------------------------------------------
		#region TransferBitsPerPixel
		private uint m_dwTransferBitsPerPixel = 0;
		private uint m_dwEnableTransferBitsPerPixel = 0;
		public uint TransferBitsPerPixel
		{
			set
			{
				if (IsOpenned())
				{
					StopTransfer();
					StTrg.SetTransferBitsPerPixel(hCamera, value);
					m_dwTransferBitsPerPixel = value;
					StartTransfer();
				}
			}
			get
			{
				if (IsOpenned())
				{
					if (m_dwTransferBitsPerPixel == 0)
					{
						StTrg.GetTransferBitsPerPixel(hCamera, out m_dwTransferBitsPerPixel);
					}
				}
				return (m_dwTransferBitsPerPixel);
			}
		}
		public uint EnableTransferBitsPerPixel
		{
			get
			{
				if (IsOpenned())
				{
					if (m_dwEnableTransferBitsPerPixel == 0)
					{
						StTrg.GetEnableTransferBitsPerPixel(hCamera, out m_dwEnableTransferBitsPerPixel);
					}
				}
				return (m_dwEnableTransferBitsPerPixel);
			}
		}
		#endregion TransferBitsPerPixel

		//---------------------------------------------------------------------------
		#region EEPROM
		public uint CameraUserNo
		{
			set
			{
				if(IsOpenned())
				{
					uint cameraNo = 0;
					StringBuilder sb = new StringBuilder(125);
					StTrg.ReadCameraUserID(hCamera, out cameraNo, sb, (uint)250);

					string cameraName = sb.ToString();
					cameraNo = value;
					StTrg.WriteCameraUserID(hCamera, cameraNo, cameraName, (uint)250);
				}
			}
			get
			{
				uint cameraNo = 0;
				if(IsOpenned())
				{
					StringBuilder sb = new StringBuilder(125);
					StTrg.ReadCameraUserID(hCamera, out cameraNo, sb, (uint)250);
				}
				return(cameraNo);
			}
		}
		public string CameraUserName
		{
			set
			{
				if(IsOpenned())
				{
					uint cameraNo = 0;
					StringBuilder sb = new StringBuilder(125);
					StTrg.ReadCameraUserID(hCamera, out cameraNo, sb, (uint)250);
					
					string cameraName = value;
					StTrg.WriteCameraUserID(hCamera, cameraNo, cameraName, (uint)250);
				}
			}
			get
			{
				string cameraName = "";
				if(IsOpenned())
				{
					uint cameraNo = 0;
					StringBuilder sb = new StringBuilder(125);
					StTrg.ReadCameraUserID(hCamera, out cameraNo, sb, (uint)250);
					cameraName = sb.ToString();
				}
				return(cameraName);
			}
		}
		#endregion

		//---------------------------------------------------------------------------
		#region Clock
		public uint ClockMode
		{
			get
			{
				uint clockMode = 0;
				uint clock = 0;
				
				if(IsOpenned())
				{
					StTrg.GetClock(hCamera, out clockMode, out clock);
				}
				return(clockMode);
			}
			set
			{
				uint clockMode = value;
				uint clock = 0;
				if(IsOpenned())
				{
					StopTransfer();
					StTrg.SetClock(hCamera, clockMode, clock);
					StartTransfer();
				}
			}
		}

		public uint PixelClock
		{
			get
			{
				uint clockMode = 0;
				uint clock = 0;
				
				if(IsOpenned())
				{
					StTrg.GetClock(hCamera, out clockMode, out clock);
				}
				return(clock);
			}
		}
		
		public uint FrameClock
		{
			get
			{
				ushort TotalLine = 0;
				ushort ClockPerLine = 0;

				if(IsOpenned())
				{
					StTrg.GetFrameClock(hCamera, out TotalLine, out ClockPerLine);
				}
				return((uint)(TotalLine * ClockPerLine));
			}
		}
		#endregion


		//---------------------------------------------------------------------------
		#region Shutter Gain Control
		public uint ExposureClock
		{
			get
			{
				uint Value = 0;
				if(IsOpenned())
				{
					StTrg.GetExposureClock(hCamera, out Value);
				}
				return(Value);
			}
			set
			{
				if(IsOpenned())
				{
					StTrg.SetExposureClock(hCamera, value);
				}
			}
		}

		public ushort TotalLine
		{
			get
			{
				ushort totalLine = 0;
				ushort clockPerLine = 0;
				StTrg.GetFrameClock(hCamera, out totalLine, out clockPerLine);
				return(totalLine);
			}
		}
		public ushort ClockPerLine
		{
			get
			{
				ushort totalLine = 0;
				ushort clockPerLine = 0;
				StTrg.GetFrameClock(hCamera, out totalLine, out clockPerLine);
				return(clockPerLine);
			}
		}
		public uint MaxExposureClock
		{
			get
			{
				ushort totalLine = 0;
				ushort clockPerLine = 0;
				StTrg.GetFrameClock(hCamera, out totalLine, out clockPerLine);
				return((uint)totalLine * clockPerLine);
			}
		}

		public string ExposureClockText
		{
			get
			{
				uint exposureClock = ExposureClock;
				uint clock = PixelClock;
				if(0 == exposureClock)
				{
					exposureClock = FrameClock;
				}
				double dblRevS = clock / (double)exposureClock;
				string strResult = String.Format("1/{0:N2} s", dblRevS);
				return(strResult);
			}

		}
		public ushort Gain
		{
			set
			{
				if(IsOpenned())
				{
					StTrg.SetGain(hCamera, value);
				}
			}
			get
			{
				ushort Value = 0;
				if(IsOpenned())
				{
					StTrg.GetGain(hCamera, out Value);
				}
				return(Value);
			}
		}

		public ushort DigitalGain
		{
			set
			{
				if(IsOpenned())
				{
					StTrg.SetDigitalGain(hCamera, value);
				}
			}
			get
			{
				ushort Value = 0;
				if(IsOpenned())
				{
					StTrg.GetDigitalGain(hCamera, out Value);
				}
				return(Value);
			}
		}
		#endregion


		//---------------------------------------------------------------------------
		#region Trigger
		public uint TriggerType
		{
			set
			{
				if(IsOpenned())
				{
					StopTransfer();

					uint triggerMode = 0;
					StTrg.GetTriggerMode(hCamera, out triggerMode);

					triggerMode &= ~StTrg.STCAM_TRIGGER_MODE_TYPE_MASK;
					triggerMode |= value & StTrg.STCAM_TRIGGER_MODE_TYPE_MASK;

					StTrg.SetTriggerMode(hCamera, triggerMode);

					StartTransfer();
				}
			}
			get
			{
				uint Value = 0;
				if(IsOpenned())
				{
					uint triggerMode = 0;
					StTrg.GetTriggerMode(hCamera, out triggerMode);

					Value = triggerMode & StTrg.STCAM_TRIGGER_MODE_TYPE_MASK;
				}
				return(Value);
			}
		}
		public uint ReadOutSource
		{
			set
			{
				if(IsOpenned())
				{
					StopTransfer();

					uint triggerMode = 0;
					StTrg.GetTriggerMode(hCamera, out triggerMode);

					triggerMode &= ~StTrg.STCAM_TRIGGER_MODE_READOUT_SOURCE_MASK;
					triggerMode |= value & StTrg.STCAM_TRIGGER_MODE_READOUT_SOURCE_MASK;

					StTrg.SetTriggerMode(hCamera, triggerMode);

					StartTransfer();
				}
			}
			get
			{
				uint Value = 0;
				if(IsOpenned())
				{
					uint triggerMode = 0;
					StTrg.GetTriggerMode(hCamera, out triggerMode);

					Value = triggerMode & StTrg.STCAM_TRIGGER_MODE_READOUT_SOURCE_MASK;
				}
				return(Value);
			}
		}
		public uint ExposureEnd
		{
			set
			{
				if(IsOpenned())
				{
					StopTransfer();

					uint triggerMode = 0;
					StTrg.GetTriggerMode(hCamera, out triggerMode);

					triggerMode &= ~StTrg.STCAM_TRIGGER_MODE_EXPEND_MASK;
					triggerMode |= value & StTrg.STCAM_TRIGGER_MODE_EXPEND_MASK;

					StTrg.SetTriggerMode(hCamera, triggerMode);

					StartTransfer();
				}
			}
			get
			{
				uint Value = 0;
				if(IsOpenned())
				{
					uint triggerMode = 0;
					StTrg.GetTriggerMode(hCamera, out triggerMode);

					Value = triggerMode & StTrg.STCAM_TRIGGER_MODE_EXPEND_MASK;
				}
				return(Value);
			}
		}
		public uint CameraMemory
		{
			set
			{
				if(IsOpenned())
				{
					StopTransfer();

					uint triggerMode = 0;
					StTrg.GetTriggerMode(hCamera, out triggerMode);

					triggerMode &= ~StTrg.STCAM_TRIGGER_MODE_CAMERA_MEMORY_MASK;
					triggerMode |= value & StTrg.STCAM_TRIGGER_MODE_CAMERA_MEMORY_MASK;

					StTrg.SetTriggerMode(hCamera, triggerMode);

					StartTransfer();
				}
			}
			get
			{
				uint Value = 0;
				if(IsOpenned())
				{
					uint triggerMode = 0;
					StTrg.GetTriggerMode(hCamera, out triggerMode);

					Value = triggerMode & StTrg.STCAM_TRIGGER_MODE_CAMERA_MEMORY_MASK;
				}
				return(Value);
			}
		}
		public uint TriggerSource
		{
			set
			{
				if(IsOpenned())
				{
					StopTransfer();

					uint triggerMode = 0;
					StTrg.GetTriggerMode(hCamera, out triggerMode);

					triggerMode &= ~StTrg.STCAM_TRIGGER_MODE_SOURCE_MASK;
					triggerMode |= value & StTrg.STCAM_TRIGGER_MODE_SOURCE_MASK;

					StTrg.SetTriggerMode(hCamera, triggerMode);

					StartTransfer();
				}
			}
			get
			{
				uint Value = 0;
				if(IsOpenned())
				{
					uint triggerMode = 0;
					StTrg.GetTriggerMode(hCamera, out triggerMode);

					Value = triggerMode & StTrg.STCAM_TRIGGER_MODE_SOURCE_MASK;
				}
				return(Value);
			}
		}
		public uint ExposureTimeType
		{
			set
			{
				if(IsOpenned())
				{
					StopTransfer();

					uint triggerMode = 0;
					StTrg.GetTriggerMode(hCamera, out triggerMode);

					triggerMode &= ~StTrg.STCAM_TRIGGER_MODE_EXPTIME_MASK;
					triggerMode |= value & StTrg.STCAM_TRIGGER_MODE_EXPTIME_MASK;

					StTrg.SetTriggerMode(hCamera, triggerMode);

					StartTransfer();
				}
			}
			get
			{
				uint Value = 0;
				if(IsOpenned())
				{
					uint triggerMode = 0;
					StTrg.GetTriggerMode(hCamera, out triggerMode);

					Value = triggerMode & StTrg.STCAM_TRIGGER_MODE_EXPTIME_MASK;
				}
				return(Value);
			}
		}
		public uint ExposureWaitHD
		{
			set
			{
				if(IsOpenned())
				{
					StopTransfer();

					uint triggerMode = 0;
					StTrg.GetTriggerMode(hCamera, out triggerMode);

					triggerMode &= ~StTrg.STCAM_TRIGGER_MODE_EXPOSURE_WAIT_HD_MASK;
					triggerMode |= value & StTrg.STCAM_TRIGGER_MODE_EXPOSURE_WAIT_HD_MASK;

					StTrg.SetTriggerMode(hCamera, triggerMode);

					StartTransfer();
				}
			}
			get
			{
				uint Value = 0;
				if(IsOpenned())
				{
					uint triggerMode = 0;
					StTrg.GetTriggerMode(hCamera, out triggerMode);

					Value = triggerMode & StTrg.STCAM_TRIGGER_MODE_EXPOSURE_WAIT_HD_MASK;
				}
				return(Value);
			}
		}
		public uint ExposureWaitReadOut
		{
			set
			{
				if(IsOpenned())
				{
					StopTransfer();

					uint triggerMode = 0;
					StTrg.GetTriggerMode(hCamera, out triggerMode);

					triggerMode &= ~StTrg.STCAM_TRIGGER_MODE_EXPOSURE_WAIT_READOUT_MASK;
					triggerMode |= value & StTrg.STCAM_TRIGGER_MODE_EXPOSURE_WAIT_READOUT_MASK;

					StTrg.SetTriggerMode(hCamera, triggerMode);

					StartTransfer();
				}
			}
			get
			{
				uint Value = 0;
				if(IsOpenned())
				{
					uint triggerMode = 0;
					StTrg.GetTriggerMode(hCamera, out triggerMode);

					Value = triggerMode & StTrg.STCAM_TRIGGER_MODE_EXPOSURE_WAIT_READOUT_MASK;
				}
				return(Value);
			}
		}
		public uint TriggerMask
		{
			set
			{
				if (IsOpenned())
				{
					StopTransfer();

					uint triggerMode = 0;
					StTrg.GetTriggerMode(hCamera, out triggerMode);

					triggerMode &= ~StTrg.STCAM_TRIGGER_MODE_TRIGGER_MASK_ON;
					triggerMode |= value & StTrg.STCAM_TRIGGER_MODE_TRIGGER_MASK_ON;

					StTrg.SetTriggerMode(hCamera, triggerMode);

					StartTransfer();
				}
			}
			get
			{
				uint Value = 0;
				if (IsOpenned())
				{
					uint triggerMode = 0;
					StTrg.GetTriggerMode(hCamera, out triggerMode);

					Value = triggerMode & StTrg.STCAM_TRIGGER_MODE_TRIGGER_MASK_ON;
				}
				return (Value);
			}
		}
		public int AutoTriggerDueTime
		{
			set
			{
				m_AutoTriggerDueTime = value;
			}
			get
			{
				return(m_AutoTriggerDueTime);
			}
		}
		public bool AutoTrigger
		{
			set
			{
				m_bAutoTrigger = value;
			}
			get
			{
				return(m_bAutoTrigger);
			}
		}
		
		public bool TriggerReadOut()
		{
			bool result = false;
			
			if(IsOpenned())
			{
				result = StTrg.TriggerReadOut(hCamera);
			}
			return(result);
		}
		
		public bool SoftTrigger()
		{
			bool result = false;
			
			if(IsOpenned())
			{
				result = StTrg.SoftTrigger(hCamera);
				if(m_bAutoTrigger)
				{
					m_objTimer.Change(m_AutoTriggerDueTime, Timeout.Infinite);
				}
			}
			return(result);
		}
		
		public bool SoftSubTrigger()
		{
			bool result = false;
			
			if(IsOpenned())
			{
				result = StTrg.SoftSubTrigger(hCamera);
			}
			return(result);
		}
		
		public bool ClearBuffer()
		{
			bool result = false;
			
			if(IsOpenned())
			{
				StopTransfer();
				result = StTrg.ClearBuffer(hCamera);
				StartTransfer();
			}
			return(result);
		}

		public bool ResetCounter()
		{
			bool result = false;
			
			if(IsOpenned())
			{
				result = StTrg.ResetCounter(hCamera);
			}
			return(result);
		}
		#endregion


		//---------------------------------------------------------------------------
		#region LED
		public uint LEDGreen
		{
			set
			{
				if(IsOpenned())
				{
					uint ledStatus = 0;
					StTrg.GetLEDStatus(hCamera, out ledStatus);

					ledStatus &= ~StTrg.STCAM_LED_GREEN_ON;
					ledStatus |= value & StTrg.STCAM_LED_GREEN_ON;

					StTrg.SetLEDStatus(hCamera, ledStatus);
				}
			}
			get
			{
				uint Value = 0;
				if(IsOpenned())
				{
					uint ledStatus = 0;
					StTrg.GetLEDStatus(hCamera, out ledStatus);

					Value = ledStatus & StTrg.STCAM_LED_GREEN_ON;
				}
				return(Value);
			}
		}
		public uint LEDRed
		{
			set
			{
				if(IsOpenned())
				{
					uint ledStatus = 0;
					StTrg.GetLEDStatus(hCamera, out ledStatus);

					ledStatus &= ~StTrg.STCAM_LED_RED_ON;
					ledStatus |= value & StTrg.STCAM_LED_RED_ON;

					StTrg.SetLEDStatus(hCamera, ledStatus);
				}
			}
			get
			{
				uint Value = 0;
				if(IsOpenned())
				{
					uint ledStatus = 0;
					StTrg.GetLEDStatus(hCamera, out ledStatus);

					Value = ledStatus & StTrg.STCAM_LED_RED_ON;
				}
				return(Value);
			}
		}
		#endregion
		//---------------------------------------------------------------------------
		#region White Balance Control
		public ushort WBRGain
		{
			set
			{
				if(IsOpenned())
				{
					ushort GainR = 0;
					ushort GainGr = 0;
					ushort GainGb = 0;
					ushort GainB = 0;
					StTrg.GetWhiteBalanceGain(hCamera, out GainR, out GainGr, out GainGb, out GainB);
					GainR = value;
					StTrg.SetWhiteBalanceGain(hCamera, GainR, GainGr, GainGb, GainB);
				}
			}
			get
			{
				ushort GainR = 0;
				ushort GainGr = 0;
				ushort GainGb = 0;
				ushort GainB = 0;
				if(IsOpenned())
				{
					StTrg.GetWhiteBalanceGain(hCamera, out GainR, out GainGr, out GainGb, out GainB);
				}
				return(GainR);
			}
		}
		
		public ushort WBGrGain
		{
			set
			{
				if(IsOpenned())
				{
					ushort GainR = 0;
					ushort GainGr = 0;
					ushort GainGb = 0;
					ushort GainB = 0;
					StTrg.GetWhiteBalanceGain(hCamera, out GainR, out GainGr, out GainGb, out GainB);
					GainGr = value;
					StTrg.SetWhiteBalanceGain(hCamera, GainR, GainGr, GainGb, GainB);
				}
			}
			get
			{
				ushort GainR = 0;
				ushort GainGr = 0;
				ushort GainGb = 0;
				ushort GainB = 0;
				if(IsOpenned())
				{
					StTrg.GetWhiteBalanceGain(hCamera, out GainR, out GainGr, out GainGb, out GainB);
				}
				return(GainGr);
			}
		}
		
		public ushort WBGbGain
		{
			set
			{
				if(IsOpenned())
				{
					ushort GainR = 0;
					ushort GainGr = 0;
					ushort GainGb = 0;
					ushort GainB = 0;
					StTrg.GetWhiteBalanceGain(hCamera, out GainR, out GainGr, out GainGb, out GainB);
					GainGb = value;
					StTrg.SetWhiteBalanceGain(hCamera, GainR, GainGr, GainGb, GainB);
				}
			}
			get
			{
				ushort GainR = 0;
				ushort GainGr = 0;
				ushort GainGb = 0;
				ushort GainB = 0;
				if(IsOpenned())
				{
					StTrg.GetWhiteBalanceGain(hCamera, out GainR, out GainGr, out GainGb, out GainB);
				}
				return(GainGb);
			}
		}
		
		public ushort WBBGain
		{
			set
			{
				if(IsOpenned())
				{
					ushort GainR = 0;
					ushort GainGr = 0;
					ushort GainGb = 0;
					ushort GainB = 0;
					StTrg.GetWhiteBalanceGain(hCamera, out GainR, out GainGr, out GainGb, out GainB);
					GainB = value;
					StTrg.SetWhiteBalanceGain(hCamera, GainR, GainGr, GainGb, GainB);
				}
			}
			get
			{
				ushort GainR = 0;
				ushort GainGr = 0;
				ushort GainGb = 0;
				ushort GainB = 0;
				if(IsOpenned())
				{
					StTrg.GetWhiteBalanceGain(hCamera, out GainR, out GainGr, out GainGb, out GainB);
				}
				return(GainB);
			}
		}
		
		public byte WhiteBalanceMode
		{
			set
			{
				if(IsOpenned())
				{
					StTrg.SetWhiteBalanceMode(hCamera, value);
				}
			}
			get
			{
				byte Value = 0;
				if(IsOpenned())
				{
					StTrg.GetWhiteBalanceMode(hCamera, out Value);
				}
				return(Value);
			}
		}
		#endregion
		//---------------------------------------------------------------------------
		#region Hue Saturaion Control
		public short Hue
		{
			set
			{
				if(IsOpenned())
				{
					byte byteHueSaturationMode = 0;
					short shtHue = 0;
					ushort wSaturation = 100;
					StTrg.GetHueSaturationMode(hCamera, out byteHueSaturationMode, out shtHue, out wSaturation);
					shtHue = value;
					StTrg.SetHueSaturationMode(hCamera, byteHueSaturationMode, shtHue, wSaturation);
				}
			}
			get
			{
				byte byteHueSaturationMode = 0;
				short shtHue = 0;
				ushort wSaturation = 100;
				if(IsOpenned())
				{
					StTrg.GetHueSaturationMode(hCamera, out byteHueSaturationMode, out shtHue, out wSaturation);
				}
				return(shtHue);
			}
		}
		public ushort Saturation
		{
			set
			{
				if(IsOpenned())
				{
					byte byteHueSaturationMode = 0;
					short shtHue = 0;
					ushort wSaturation = 100;
					StTrg.GetHueSaturationMode(hCamera, out byteHueSaturationMode, out shtHue, out wSaturation);
					wSaturation = value;
					StTrg.SetHueSaturationMode(hCamera, byteHueSaturationMode, shtHue, wSaturation);
				}
			}
			get
			{
				byte byteHueSaturationMode = 0;
				short shtHue = 0;
				ushort wSaturation = 100;
				if(IsOpenned())
				{
					StTrg.GetHueSaturationMode(hCamera, out byteHueSaturationMode, out shtHue, out wSaturation);
				}
				return(wSaturation);
			}
		}
		public byte HueSaturationMode
		{
			set
			{
				if(IsOpenned())
				{
					byte byteHueSaturationMode = 0;
					short shtHue = 0;
					ushort wSaturation = 100;
					StTrg.GetHueSaturationMode(hCamera, out byteHueSaturationMode, out shtHue, out wSaturation);
					byteHueSaturationMode = value;
					StTrg.SetHueSaturationMode(hCamera, byteHueSaturationMode, shtHue, wSaturation);
				}
			}
			get
			{
				byte byteHueSaturationMode = 0;
				short shtHue = 0;
				ushort wSaturation = 100;
				if(IsOpenned())
				{
					StTrg.GetHueSaturationMode(hCamera, out byteHueSaturationMode, out shtHue, out wSaturation);
				}
				return(byteHueSaturationMode);
			}
		}
		#endregion

		//---------------------------------------------------------------------------
		#region TimeOut
		public bool SetTimeOut(uint dwTimeOutType, uint dwTimeOutMS)
		{
			bool result = false;
			
			if(IsOpenned())
			{
				result = StTrg.SetTimeOut(hCamera, dwTimeOutType, dwTimeOutMS);
			}
			return(result);
		}
		public bool GetTimeOut(uint dwTimeOutType, out uint dwTimeOutMS)
		{
			bool result = false;
			dwTimeOutMS = 0;
			if(IsOpenned())
			{
				result = StTrg.GetTimeOut(hCamera, dwTimeOutType, out dwTimeOutMS);
			}
			return(result);
		}

		#endregion


		//---------------------------------------------------------------------------
		#region Save Image
		public bool SaveImage(uint dwWidth, uint dwHeight, uint dwPreviewPixelFormat, System.IntPtr pbyteData, string pszFileName, uint dwParam)
		{
			bool result = false;
			
			if(IsOpenned())
			{
				result = StTrg.SaveImage(dwWidth, dwHeight, dwPreviewPixelFormat, pbyteData, pszFileName, dwParam);
			}
			return(result);
		}
		#endregion


		//---------------------------------------------------------------------------
		#region Setting
		private bool SettingFilePlus(string fileName, bool isRead)
		{
			bool result = true;

			string appName = string.Format("Other-{0:X4}", usbProductID);
			string keyName = "ColorInterpolationMethod";
			if(isRead)
			{
				uint Value = colorInterpolationMethod;
				Value = Native.GetPrivateProfileInt(appName, keyName, (int)Value, fileName);
				colorInterpolationMethod = (byte)Value;
			}
			else
			{
				string Value = colorInterpolationMethod.ToString();
				Native.WritePrivateProfileString(appName, keyName, Value, fileName);

			}

			keyName = "MirrorMode";
			if(isRead)
			{
				uint Value = mirrorMode;
				Value = Native.GetPrivateProfileInt(appName, keyName, (int)Value, fileName);
				mirrorMode = (byte)Value;
			}
			else
			{
				string Value = mirrorMode.ToString();
				Native.WritePrivateProfileString(appName, keyName, Value, fileName);

			}
			keyName = "RotationMode";
			if(isRead)
			{
				uint Value = rotationMode;
				Value = Native.GetPrivateProfileInt(appName, keyName, (int)Value, fileName);
				rotationMode = (byte)Value;
			}
			else
			{
				string Value = rotationMode.ToString();
				Native.WritePrivateProfileString(appName, keyName, Value, fileName);
			}

			if (IsJBA)
			{
				appName = "miniMMLSampleViewer";
				keyName = "LED1_ON";
				if (isRead)
				{
					uint Value = LED1_ON;
					Value = Native.GetPrivateProfileInt(appName, keyName, (int)Value, fileName);
					LED1_ON = (ushort)Value;
				}
				else
				{
					string Value = LED1_ON.ToString();
					Native.WritePrivateProfileString(appName, keyName, Value, fileName);
				}

				keyName = "LED2_ON";
				if (isRead)
				{
					uint Value = LED2_ON;
					Value = Native.GetPrivateProfileInt(appName, keyName, (int)Value, fileName);
					LED2_ON = (ushort)Value;
				}
				else
				{
					string Value = LED2_ON.ToString();
					Native.WritePrivateProfileString(appName, keyName, Value, fileName);
				}

				keyName = "LED1_Power";
				if (isRead)
				{
					uint Value = LED1_Power;
					Value = Native.GetPrivateProfileInt(appName, keyName, (int)Value, fileName);
					LED1_Power = (ushort)Value;
				}
				else
				{
					string Value = LED1_Power.ToString();
					Native.WritePrivateProfileString(appName, keyName, Value, fileName);
				}

				keyName = "LED2_Power";
				if (isRead)
				{
					uint Value = LED2_Power;
					Value = Native.GetPrivateProfileInt(appName, keyName, (int)Value, fileName);
					LED2_Power = (ushort)Value;
				}
				else
				{
					string Value = LED2_Power.ToString();
					Native.WritePrivateProfileString(appName, keyName, Value, fileName);
				}

				keyName = "HDRMode";
				if (isRead)
				{
					uint Value = HDRMode;
					Value = Native.GetPrivateProfileInt(appName, keyName, (int)Value, fileName);
					HDRMode = (ushort)Value;
				}
				else
				{
					string Value = HDRMode.ToString();
					Native.WritePrivateProfileString(appName, keyName, Value, fileName);
				}

				keyName = "T2Ratio";
				if (isRead)
				{
					uint Value = T2Ratio;
					Value = Native.GetPrivateProfileInt(appName, keyName, (int)Value, fileName);
					T2Ratio = (ushort)Value;
				}
				else
				{
					string Value = T2Ratio.ToString();
					Native.WritePrivateProfileString(appName, keyName, Value, fileName);
				}

				keyName = "T3Ratio";
				if (isRead)
				{
					uint Value = T3Ratio;
					Value = Native.GetPrivateProfileInt(appName, keyName, (int)Value, fileName);
					T3Ratio = (ushort)Value;
				}
				else
				{
					string Value = T3Ratio.ToString();
					Native.WritePrivateProfileString(appName, keyName, Value, fileName);
				}

				keyName = "V1Step";
				if (isRead)
				{
					uint Value = V1Step;
					Value = Native.GetPrivateProfileInt(appName, keyName, (int)Value, fileName);
					V1Step = (ushort)Value;
				}
				else
				{
					string Value = V1Step.ToString();
					Native.WritePrivateProfileString(appName, keyName, Value, fileName);
				}

				keyName = "V2Step";
				if (isRead)
				{
					uint Value = V2Step;
					Value = Native.GetPrivateProfileInt(appName, keyName, (int)Value, fileName);
					V2Step = (ushort)Value;
				}
				else
				{
					string Value = V2Step.ToString();
					Native.WritePrivateProfileString(appName, keyName, Value, fileName);
				}

				keyName = "V3Step";
				if (isRead)
				{
					uint Value = V3Step;
					Value = Native.GetPrivateProfileInt(appName, keyName, (int)Value, fileName);
					V2Step = (ushort)Value;
				}
				else
				{
					string Value = V3Step.ToString();
					Native.WritePrivateProfileString(appName, keyName, Value, fileName);
				}

				keyName = "ADCMode";
				if (isRead)
				{
					uint Value = ADCMode;
					Value = Native.GetPrivateProfileInt(appName, keyName, (int)Value, fileName);
					V2Step = (ushort)Value;
				}
				else
				{
					string Value = ADCMode.ToString();
					Native.WritePrivateProfileString(appName, keyName, Value, fileName);
				}
			}

			return(result);
		}
		public bool WriteSettingFile(string fileName)
		{
			bool result = false;
			
			if(IsOpenned())
			{
				StopTransfer();
				result = StTrg.WriteSettingFile(hCamera, fileName);
				result = SettingFilePlus(fileName, false);
				StartTransfer();
			}
			return(result);
		}
		public bool ReadSettingFile(string fileName)
		{
			bool result = false;
			if(IsOpenned())
			{
				StopTransfer();
				result = StTrg.ReadSettingFile(hCamera, fileName);
				result = SettingFilePlus(fileName, true);
				StartTransfer();
			}
			return(result);
		}
		#endregion

		//---------------------------------------------------------------------------
		#region Sharpness
		public byte SharpnessMode
		{
			set
			{

				if(IsOpenned())
				{
					byte sharpnessMode = 0;
					ushort sharpnessGain = 0;
					byte sharpnessCoring = 0;
					StTrg.GetSharpnessMode(hCamera, out sharpnessMode, out sharpnessGain, out sharpnessCoring);
					sharpnessMode = value;
					StTrg.SetSharpnessMode(hCamera, sharpnessMode, sharpnessGain, sharpnessCoring);
				}
			}
			get
			{
				byte Value = 0;
				if(IsOpenned())
				{
					byte sharpnessMode = 0;
					ushort sharpnessGain = 0;
					byte sharpnessCoring = 0;
					StTrg.GetSharpnessMode(hCamera, out sharpnessMode, out sharpnessGain, out sharpnessCoring);
					Value = sharpnessMode;
				}
				return(Value);
			}
		}
		public ushort SharpnessGain
		{
			set
			{

				if(IsOpenned())
				{
					byte sharpnessMode = 0;
					ushort sharpnessGain = 0;
					byte sharpnessCoring = 0;
					StTrg.GetSharpnessMode(hCamera, out sharpnessMode, out sharpnessGain, out sharpnessCoring);
					sharpnessGain = value;
					StTrg.SetSharpnessMode(hCamera, sharpnessMode, sharpnessGain, sharpnessCoring);
				}
			}
			get
			{
				ushort Value = 0;
				if(IsOpenned())
				{
					byte sharpnessMode = 0;
					ushort sharpnessGain = 0;
					byte sharpnessCoring = 0;
					StTrg.GetSharpnessMode(hCamera, out sharpnessMode, out sharpnessGain, out sharpnessCoring);
					Value = sharpnessGain;
				}
				return(Value);
			}
		}
		public byte SharpnessCoring
		{
			set
			{

				if(IsOpenned())
				{
					byte sharpnessMode = 0;
					ushort sharpnessGain = 0;
					byte sharpnessCoring = 0;
					StTrg.GetSharpnessMode(hCamera, out sharpnessMode, out sharpnessGain, out sharpnessCoring);
					sharpnessCoring = value;
					StTrg.SetSharpnessMode(hCamera, sharpnessMode, sharpnessGain, sharpnessCoring);
				}
			}
			get
			{
				byte Value = 0;
				if(IsOpenned())
				{
					byte sharpnessMode = 0;
					ushort sharpnessGain = 0;
					byte sharpnessCoring = 0;
					StTrg.GetSharpnessMode(hCamera, out sharpnessMode, out sharpnessGain, out sharpnessCoring);
					Value = sharpnessCoring;
				}
				return(Value);
			}
		}
		#endregion

		#region JBA
		public bool IsJBA
		{
			get
			{
				bool value = false;
				switch (usbProductID)
				{
					case (StTrg.STCAM_USBPID_APBWVUSB_LED):
					case (StTrg.STCAM_USBPID_APCWVUSB_LED):
						value = true;
						break;
				}
				return (value);
			}
		}
		public ushort LED1_ON
		{
			get
			{
				return (IsLEDOn(0));
			}
			set
			{
				SetLEDOn(0, value);
			}
		}
		public ushort LED2_ON
		{
			get
			{
				return (IsLEDOn(1));
			}
			set
			{
				SetLEDOn(1, value);
			}
		}
		public ushort LED1_Power
		{
			get
			{
				return (GetLEDPower(0));
			}
			set
			{
				SetLEDPower(0, value);
			}
		}
		public ushort LED2_Power
		{
			get
			{
				return (GetLEDPower(1));
			}
			set
			{
				SetLEDPower(1, value);
			}
		}
		private ushort IsLEDOn(byte ledNo)
		{
			ushort value = 0;
			byte byteStartAdd = (byte)(StTrg.JBA_REG_ADD_LED1 + ledNo);
			if (0 != (GetRegister(byteStartAdd) & 0x0080))
			{
				value = 1;
			}
			return (value);
		}
		private void SetLEDOn(byte ledNo, ushort ledOn)
		{
			if (!IsJBA) return;
			ushort value = GetLEDPower(ledNo);
			if (ledOn != 0)
			{
				value |= 0x0080;
			}

			byte byteStartAdd = (byte)(StTrg.JBA_REG_ADD_LED1 + ledNo);
			byte byteEndAdd = byteStartAdd;
			SetRegister(byteStartAdd, value);
		}
		private ushort GetLEDPower(byte ledNo)
		{
			byte byteStartAdd = (byte)(StTrg.JBA_REG_ADD_LED1 + ledNo);
			return ((ushort)(GetRegister(byteStartAdd) & 0x007F));
		}
		private void SetLEDPower(byte ledNo, ushort power)
		{
			if (!IsJBA) return;
			ushort value = power;
			if (IsLEDOn(ledNo) != 0)
			{
				value |= 0x0080;
			}

			byte byteStartAdd = (byte)(StTrg.JBA_REG_ADD_LED1 + ledNo);
			SetRegister(byteStartAdd, value);
		}

		public ushort HDRMode
		{
			get { return (GetRegister(StTrg.JBA_REG_ADD_HDR_MODE)); }
			set { SetRegister(StTrg.JBA_REG_ADD_HDR_MODE, value); }
		}
		public ushort T2Ratio
		{
			get { return (GetRegister(StTrg.JBA_REG_ADD_T2_RATIO)); }
			set { SetRegister(StTrg.JBA_REG_ADD_T2_RATIO, value); }
		}
		public ushort T3Ratio
		{
			get { return (GetRegister(StTrg.JBA_REG_ADD_T3_RATIO)); }
			set { SetRegister(StTrg.JBA_REG_ADD_T3_RATIO, value); }
		}
		public ushort V1Step
		{
			get { return (GetRegister(StTrg.JBA_REG_ADD_V1_STEP)); }
			set { SetRegister(StTrg.JBA_REG_ADD_V1_STEP, value); }
		}
		public ushort V2Step
		{
			get { return (GetRegister(StTrg.JBA_REG_ADD_V2_STEP)); }
			set { SetRegister(StTrg.JBA_REG_ADD_V2_STEP, value); }
		}
		public ushort V3Step
		{
			get { return (GetRegister(StTrg.JBA_REG_ADD_V3_STEP)); }
			set { SetRegister(StTrg.JBA_REG_ADD_V3_STEP, value); }
		}
		public ushort ADCMode
		{
			get { return (GetRegister(StTrg.JBA_REG_ADD_ADC_MODE)); }
			set { SetRegister(StTrg.JBA_REG_ADD_ADC_MODE, value); }
		}
		public ushort GetRegister(uint nAdd)
		{
			ushort[] pwValue = new ushort[] { 0x0000 };

			unsafe
			{
				fixed (ushort* ptr = &pwValue[0])
				{
					bool bResult = StTrg.GetJBARegister(hCamera, nAdd, nAdd, (IntPtr)ptr);
				}
			}
			
			return (pwValue[0]);
		}
		public void SetRegister(uint nAdd, ushort value)
		{
			ushort[] pwValue = new ushort[] { value };
			unsafe
			{
				fixed (ushort* ptr = &pwValue[0])
				{
					bool bResult = StTrg.SetJBARegister(hCamera, nAdd, nAdd, (IntPtr)ptr);
				}
			}
		}
		#endregion
	}
}
